define([
    'underscore',
    'backbone',
    'App',
    'Layout',
    'models/facilities-collection',
    'models/patient-identifiers-collection',
    'modules/community-care-request/new-appointment-request-question/facility-caretype-selection/community-care-facility-caretype-layout-view',
    'modules/community-care-request/new-appointment-request-question/facility-caretype-selection/caretype-section-view',
    'modules/community-care-request/new-appointment-request-question/facility-caretype-selection/facility-section-view',
],
function (
    _,
    Backbone,
    app,
    Layout,
    Facilities,
    PatientIdentifiers,
    CCFacilityCareTypeLayoutView,
    CareTypeSectionView,
    FacilitySectionView
) {
    'use strict';

    describe('Facility-CareType Layout', function () {
        var view;
        var $view;
        var layout = new Layout();
        var model;
        var collection;

        layout.render();

        afterEach(function () {
            layout.getRegion('content').empty();
        });

        beforeEach(function() {
            app.patientIdentifiers = new PatientIdentifiers();
            app.patientIdentifiers.add(new Backbone.Model({ assigningAuthority: 'dfn-100', uniqueId: '100001' }));
            app.facilities = new Facilities([
                { facilityCode: '100', name: 'One VA Facility', supportsVAR: false },
                { facilityCode: '200', name: 'Two VA Facility', supportsVAR: false },
            ]);
            app.ccTypeOfCares = new Backbone.Collection([
                {
                    'id': 'CCAUDHEAR',
                    'name': 'Audiology (hearing aid support)',
                    'details': 'This is an office visit for Veterans who already have a hearing aid and who need assistance with the device. This visit is for trouble shooting or adjusting your hearing aid for best performance. This may include technical adjustments, cleaning, and/or reprogramming for new listening situations and environments. This is not a visit to initially be evaluated for or to obtain a hearing aid.',
                    'confirmation': 'This request is for hearing aid support only. If you have other needs, please contact your Primary Care Provider to discuss the issue. If you are distance eligible and do not have a Primary Care Provider, you can use this app to request help in getting one assigned.',
                },
                {
                    'id': 'CCAUDRTNE',
                    'name': 'Audiology (routine hearing exam)',
                    'details': 'This includes an office visit for a hearing exam and evaluation using non-invasive tests to check your hearing and inner ear health. The exam may lead to prevention and treatment recommendations, a request for further consultation, and/or a recommendation for hearing aids. A routine exam is not for any new or sudden changes with your hearing or ears.',
                    'confirmation': 'This request is for routine hearing exam only. If you have other needs, please contact your Primary Care Provider to discuss the issue. If you are distance eligible and do not have a Primary Care Provider, you can use this app to request help in getting one assigned.',
                },
                {
                    'id': 'CCOPT',
                    'name': 'Optometry (routine eye exam)',
                    'details': 'This includes an office visit for an eye exam and vision evaluation, which may include dilating your eyes. The exam may lead to prevention and treatment recommendations, a request for further consultation, and/or a prescription for eye glasses or contact lenses. A routine exam is not for any new or sudden changes with your vision or eyes.',
                    'confirmation': 'This request currently only supports routine eye exams. If you have other needs, please contact your Primary Care Provider to discuss the issue. If you are distance eligible and do not have a Primary Care Provider, you can use this app to request help in getting one assigned.',
                },
            ]);
        });


        describe('show facility and caretype layout view', function () {
            collection = new Backbone.Collection([{ facilityCode: '100', name: 'One VA Facility', supportsVAR: false }]);
            model = new Backbone.Model({steps: [3]});

            beforeEach(function () {
                spyOn(CCFacilityCareTypeLayoutView.prototype, 'routeToPrevious');
                spyOn(CCFacilityCareTypeLayoutView.prototype, 'showChildView');
                view = new CCFacilityCareTypeLayoutView({
                    model: model,
                    validator: {
                        checkForm: function() {
                            return true;
                        },
                    },
                });
                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('has regions to show section views', function () {
                expect($view.has(view.regions.facilitySelectionSection).length).toEqual(1);
                expect($view.has(view.regions.careTypeSelectionSection).length).toEqual(1);
            });

            it('show the facility and caretype section view initials', function () {
                var args = view.showChildView.calls.allArgs();

                expect(view.showChildView).toHaveBeenCalled();
                expect(args[0][0]).toBe('facilitySelectionSection');
                expect(args[1][0]).toBe('careTypeSelectionSection');
            });
            it('has a button to the previous page', function() {
                var backButton = $view.find('#previous-btn');
                backButton.click();
                expect(view.routeToPrevious).toHaveBeenCalled();
            });
        });

        describe('show type of care for patient with single DFN', function () {
            model = new Backbone.Model({steps: [3]});

            beforeEach(function () {
                view = new CareTypeSectionView({
                    model: model,
                    ccTypeOfCares: app.ccTypeOfCares,
                });
                layout.showChildView('content', view);
                $view = view.$el;
                
            });

            it('shows caretype options as drop-down list', function () {
                expect(view.collection.get('careType').get('type')).toEqual('select');
            });

            it('shows caretype details at a selected caretype option', function () {
                expect(view.collection.get('careTypeDetails').get('type')).toEqual('static');
            });

            it('shows continue-on button at a selected caretype option', function () {
                expect(view.collection.get('careTypeContinue').get('type')).toEqual('radio');
            });
        });

        describe('show facility section view for patient with multi-DFNs', function () {
            app.patientIdentifiers.add(new Backbone.Model({ assigningAuthority: 'dfn-200', uniqueId: '200001' }));
            collection = new Backbone.Collection([
                { facilityCode: '100', name: 'One VA Facility', supportsVAR: false },
                { facilityCode: '200', name: 'Two VA Facility', supportsVAR: true },
            ]);
            model = new Backbone.Model({steps: [3]});

            beforeEach(function () {
                view = new FacilitySectionView({
                    model: model,
                    patientEnrolledFacilities: collection,
                    hasMultiFacilities: true,
                });
                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('shows facility options as radio buttons', function () {
                expect(view.collection.get('facility').get('type')).toEqual('radio');
            });

            it('select a facility with facility code = 100', function () {
                expect($view.find('label[for="facility-100"] .label-text > span').text()).toEqual('One VA Facility');
                expect($view.find('label[for="facility-100"] .label-details p').text()).toEqual('This location is not currently supporting community care requests.');
            });

            it('select a facility with facility code = 200', function () {
                expect($view.find('label[for="facility-200"] .label-text > span').text()).toEqual('Two VA Facility');
            });
        });
    });
}
);
